<?php

namespace App\Http\Controllers\Admin\Blog;

use App\Http\Controllers\Controller;
use App\Http\Requests\StoreBlogRequest;
use App\Http\Requests\UpdateBlogRequest;
use App\Models\Blog;
use App\Models\Category;
use App\Models\Destination;
use Illuminate\Support\Str;

class BlogController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $blogs = Blog::with(['category', 'destination'])->latest()->get();

        return view('admin.blog.index', compact('blogs'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $blog = new Blog;
        $categories = Category::getType('blog')->select('name', 'id')->get();
        $destinations = Destination::where('status', true)->select('name', 'id')->get();

        return view('admin.blog.create', compact('blog', 'categories', 'destinations'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreBlogRequest $request)
    {
        $data = $request->validated();
        $data['status'] = $request->has('status');
        $data['slug'] = Str::slug($data['title']);

        if (user()->hasRole('agent')) {
            $data['agent_id'] = user()->id;
        }
        // if (Blog::where('slug', $data['slug'])->where('destination_id', $data['destination_id'])->where('category_id', $data['category_id'])->exists()) {
        //     return redirect()->back()->with('error', 'Blog with this name already exists.');
        // }

        try {
            $blog = Blog::updateOrCreate([
                'slug' => $data['slug'],
                'destination_id' => $data['destination_id'],
                'category_id' => $data['category_id'],
            ], $data);
            if ($request->has('thumbnail')) {
                $blog->image()->create([
                    'path' => $request->file('thumbnail')->store('blogs', 'public'),
                    'type' => 'thumbnail',
                ]);
            }

            return redirect()->route('admin.blog.index')->with('success', 'Blog created successfully.');
        } catch (\Exception $e) {
            return $e->getMessage();

            return redirect()->back()->with('error', 'Blog could not be created.');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Blog $blog)
    {
        abort(404);

        return view('admin.blog.show', compact('blog'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Blog $blog)
    {
        $categories = Category::getType('blog')->select('name', 'id')->get();
        $destinations = Destination::where('status', true)->select('name', 'id')->get();

        return view('admin.blog.create', compact('blog', 'categories', 'destinations'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateBlogRequest $request, Blog $blog)
    {
        $data = $request->validated();
        $data['status'] = $request->has('status');
        $data['slug'] = Str::slug($data['title']);

        // if (Blog::where('slug', $data['slug'])->where('destination_id', $data['destination_id'])->where('category_id', $data['category_id'])->where('id', '!=', $blog->id)->exists()) {
        //     return redirect()->back()->with('error', 'Blog with this name already exists.');
        // }

        try {
            $blog->update($data);
            if ($request->has('thumbnail')) {
                removeFile(public_path('storage/'.$blog->thumbnail));
                $blog->images()->where('type', 'thumbnail')->delete();
                $blog->image()->create([
                    'path' => $request->file('thumbnail')->store('blogs', 'public'),
                    'type' => 'thumbnail',
                ]);
            }

            return redirect()->route('admin.blog.index')->with('success', 'Blog updated successfully.');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Blog could not be updated.');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Blog $blog)
    {
        try {
            removeFile(public_path('storage/'.$blog->thumbnail));
            $blog->delete();

            return response()->json(['message' => 'Blog deleted successfully.', 'status' => 'success']);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Blog could not be deleted.', 'status' => 'error'], 500);
        }
    }
}
